﻿#pragma once
#include "CommonStates.h"
#include "SpriteBatch.h"
#include "RendererDefines.h"

namespace RTCam {

class AppWindow;

// Helper class that initializes DirectX APIs for 3D rendering.
class Direct3DBase abstract
{
public:
	Direct3DBase(void);
	~Direct3DBase(void);

	virtual void Initialize(const shared_ptr<AppWindow>& window);	// Initialize the Direct3D resources required to run.
	virtual void CheckCapabilities();								// Check whether the required DirectX capabilities are supported
	virtual void HandleDeviceLost();								// Restore all device resources and set them back to the current state
	virtual void CreateDeviceResources();							// These are the resources that depend on the device.
	virtual void CreateWindowSizeDependentResources();				// Allocate all memory resources that change on a window SizeChanged event.
	virtual void UpdateForWindowSizeChange();						// This method is called in the event handler for the SizeChanged event.
	virtual void Render(const shared_ptr<class Scene>& scene) = 0;										
	virtual void Present();											// Method to deliver the final image to the display.

	// Direct3D objects.
	ComPtr<ID3D11Device1>			m_d3dDevice;		// Used to create resources.
	ComPtr<ID3D11DeviceContext1>	m_deviceContext;	// Used to perform rendering.
	ComPtr<IDXGISwapChain1>			m_swapChain;		// Holds the buffers to which the device renders.

	// Direct2D and DirectWrite objects.
	ComPtr<ID2D1Factory>			m_d2dFactory;
	ComPtr<ID2D1RenderTarget>		m_d2dBackbufferRT;
	ComPtr<IDWriteFactory>			m_dWriteFactory;
	ComPtr<IDWriteTextFormat>		m_defaultTextFormat;
	ComPtr<ID2D1SolidColorBrush>	m_blackBrush;
	ComPtr<ID2D1SolidColorBrush>	m_whiteBrush;

	ComPtr<ID3DUserDefinedAnnotation> m_userAnnotations;

	static const float RecommendedLineSpacing;

protected:

	virtual void ReleaseResources();					// Release COM objects (other than the swap chain)
	
	void CreateD2DWindowSizeDependentResources();
	void ReleaseD2DWindowSizeDependentResources();

	// DirectX Helpers
	unique_ptr<DirectX::CommonStates> m_states;
	unique_ptr<DirectX::SpriteBatch> m_spriteBatch;

	// The rendertarget that gets displayed.
	ComPtr<ID3D11RenderTargetView> m_finalRenderTargetView;
	ComPtr<ID3D11DepthStencilView> m_finalDepthStencilView;

	// Cached renderer properties.
	D3D_FEATURE_LEVEL m_featureLevel;
	DirectX::XMINT2 m_renderSize;
	weak_ptr<AppWindow> m_window;
};

}